#!/usr/bin/env ruby

require "matrix"

filename = "hilbert_cube.scad"

center = Vector[0, 0, 0] # where to build around
side = 10 # how far between each point in the path?
iterations = 1 # only 0 - 2 anything higher is just crazy

case iterations
when 0
  block_size = side / 2.0
when 1
  block_size = side / 4.0
when 2
  block_size = side / 8.0
else
  raise "Please use an iteration between 0 - 2 unless you have a supercomputer"
end

# figure out a block size that results in a solid path
# side, iterations = block_size
# 20 , 0 = 10 = 20/2
# 20 , 1 = 5 = 20/4
# 20 , 2 = 2.5 = 20/8
# block_size = 2.5

# when side = 10:
# 0 iterations = 15 cube = 1.5x
# 1 iterations = 17.5 cube = 1.75x
# 2 iterations = 18.75 cube = 1.875x

# handy shortcuts
class Vector
  def x
    self[0]
  end

  def y
    self[1]
  end

  def z
    self[2]
  end
  
  def join(seperator)
    "#{self[0]}#{seperator}#{self[1]}#{seperator}#{self[2]}"
  end
end

# hilbert algorithm cargo culted from: http://www.openprocessing.org/visuals/?visualID=15599 by Thomas Diewald
def hilbert_3d(center, side, iterations, v0, v1, v2, v3, v4, v5, v6, v7)
  vec_s = [
    Vector[  (center.x - side/2.0),   (center.y + side/2.0), (center.z - side/2.0)  ],
    Vector[  (center.x - side/2.0),   (center.y + side/2.0), (center.z + side/2.0)  ],
    Vector[  (center.x - side/2.0),   (center.y - side/2.0), (center.z + side/2.0)  ],
    Vector[  (center.x - side/2.0),   (center.y - side/2.0), (center.z - side/2.0)  ],
    Vector[  (center.x + side/2.0),   (center.y - side/2.0), (center.z - side/2.0)  ],
    Vector[  (center.x + side/2.0),   (center.y - side/2.0), (center.z + side/2.0)  ],
    Vector[  (center.x + side/2.0),   (center.y + side/2.0), (center.z + side/2.0)  ],
    Vector[  (center.x + side/2.0),   (center.y + side/2.0), (center.z - side/2.0)  ]
  ];

  vec = [ vec_s[ v0 ], vec_s[ v1 ], vec_s[ v2 ], vec_s[ v3 ], vec_s[ v4 ], vec_s[ v5 ], vec_s[ v6 ], vec_s[ v7 ] ];

  if (iterations -1 >= 0) then
    tmp = [];
    tmp.concat(hilbert_3d( vec[ 0 ],  side/2.0,   iterations - 1,   v0, v3, v4, v7, v6, v5, v2, v1  ));
    tmp.concat(hilbert_3d( vec[ 1 ],  side/2.0,   iterations - 1,   v0, v7, v6, v1, v2, v5, v4, v3  ));
    tmp.concat(hilbert_3d( vec[ 2 ],  side/2.0,   iterations - 1,   v0, v7, v6, v1, v2, v5, v4, v3  ));
    tmp.concat(hilbert_3d( vec[ 3 ],  side/2.0,   iterations - 1,   v2, v3, v0, v1, v6, v7, v4, v5  ));
    tmp.concat(hilbert_3d( vec[ 4 ],  side/2.0,   iterations - 1,   v2, v3, v0, v1, v6, v7, v4, v5  ));
    tmp.concat(hilbert_3d( vec[ 5 ],  side/2.0,   iterations - 1,   v4, v3, v2, v5, v6, v1, v0, v7  ));
    tmp.concat(hilbert_3d( vec[ 6 ],  side/2.0,   iterations - 1,   v4, v3, v2, v5, v6, v1, v0, v7  ));
    tmp.concat(hilbert_3d( vec[ 7 ],  side/2.0,   iterations - 1,   v6, v5, v2, v1, v0, v3, v4, v7  ));
    return tmp
  else
    return vec
  end
end

vectors = hilbert_3d(center, side, iterations, 0, 1, 2, 3, 4, 5, 6, 7)

puts vectors.inspect
puts vectors.size

scad_file = File.open(filename, "w")

scad_file << "union() {\n"
vectors.each_with_index do |vector, i|
  previous_vector = vectors[i-1]

  scad_file << "translate([#{vector.join(',')}]) cube(#{block_size}, center=true);\n"

  if i > 0 and vector != previous_vector then
    # find midpoint between this and last 
    middle_vector = Vector[(vector.x + previous_vector.x) / 2, (vector.y + previous_vector.y) / 2, (vector.z + previous_vector.z) / 2]
    puts middle_vector

    # find direction
    # if previous_vector.x != vector.x then
    #   connecting_cube_dimensions = [block_size * 4, block_size/2, block_size/2]
    # elsif previous_vector.y != vector.y then
    #   connecting_cube_dimensions = [block_size/2, block_size * 4, block_size/2]
    # elsif previous_vector.z != vector.z then
    #   connecting_cube_dimensions = [block_size/2, block_size/2, block_size * 4]
    # else
    #   raise "Couldn't find connecting cube direction?"
    # end
    # scad_file << "color([1,1,1]) translate([#{middle_vector.join(',')}]) cube([#{connecting_cube_dimensions.join(',')}], center=true);\n"
    
    scad_file << "color([1,1,1]) translate([#{middle_vector.join(',')}]) cube(#{block_size}, center=true);\n"
  end
end
scad_file << "}\n"

scad_file.close
